/*:
 * @target MZ
 * @plugindesc Backlog Close Input Flush — バックログを閉じた直後の入力トリガを数フレーム消して誤反応を防止（Sprite_Pictureは触りません）
 * @author HS
 *
 * @param flushFrames
 * @text 入力クリアフレーム数
 * @type number
 * @min 1
 * @max 10
 * @default 2
 * @desc バックログを閉じた後、このフレーム数だけ Input / TouchInput を clear します（60fpsで2=約33ms）。
 *
 * @help
 * - Sprite_Picture の isClickEnabled/onClick を上書きしないため、ButtonPicture と干渉しにくい安全版です。
 * - 目的は「バックログを閉じたクリックが背面UIに貫通する」の抑止のみ。
 * - HS_BacklogCloseClickDebounce.js の代替として使用してください（併用しない）。
 */
(() => {
  "use strict";

  const PN = document.currentScript.src.split("/").pop().replace(/\.js$/, "");
  const P  = PluginManager.parameters(PN);
  const FRAMES = Math.max(1, Math.min(10, +P.flushFrames || 2));

  const isBacklogOpen = () => !!(window.HSBL && typeof HSBL.isOpen === "function" && HSBL.isOpen());

  let prevOpen = false;
  let flushLeft = 0;

  function resetState() {
    flushLeft = 0;
    prevOpen = isBacklogOpen();
  }

  // Sceneごとの監視：閉じた瞬間に数フレーム入力を掃除
  const _SceneBase_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function() {
    _SceneBase_update.apply(this, arguments);

    const nowOpen = isBacklogOpen();
    if (prevOpen && !nowOpen) {
      flushLeft = FRAMES;
    }
    prevOpen = nowOpen;

    if (flushLeft > 0) {
      flushLeft--;
      try { Input.clear?.(); TouchInput.clear?.(); } catch(_) {}
      // 一部環境でclearだけだと押下状態が残る保険
      try { TouchInput._mousePressed = false; } catch(_) {}
    }
  };

  // ロード/タイトル復帰などでも保険で状態を初期化
  const _extract = DataManager.extractSaveContents;
  DataManager.extractSaveContents = function(contents) {
    _extract.apply(this, arguments);
    resetState();
  };

  if (Scene_Load?.prototype?.onLoadSuccess) {
    const _onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
    Scene_Load.prototype.onLoadSuccess = function() {
      _onLoadSuccess.apply(this, arguments);
      resetState();
    };
  }

  // デバッグ用（任意）
  window.HS = window.HS || {};
  HS.isBacklogCloseInputFlushing = () => flushLeft > 0;
})();
